<?php
/**
 * Plugin Name: AppHaven Font Resizer
 * Plugin URI:  https://apphaven.net/plugins/apphaven-font-resizer
 * Description: Adds an accessible font-size widget with Normal, Large, and X-Large options to resize the current post/page text.
 * Version:     1.0.0
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * Author:      Michael Grant / AppHaven
 * Author URI:  https://apphaven.net
 * Tested up to: 6.8.3
 * License:     GPL-2.0+
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: apphaven-font-resizer
 * Domain Path: /languages
 *
 * @package AppHaven_Font_Resizer
 */

if (!defined('ABSPATH')) {
    exit;
}

class AppHaven_Font_Resizer {

    const HANDLE_JS  = 'ah-font-resizer';
    const HANDLE_CSS = 'ah-font-resizer-css';

    public function __construct() {
        // Front assets
        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);

        // Wrap the main content so we can scale it safely
        add_filter('the_content', [$this, 'wrap_post_content'], 9);

        // Register widget
        add_action('widgets_init', function () {
            register_widget('AppHaven_Font_Resizer_Widget');
        });
    }

    public function enqueue_assets() {
        if (!is_singular()) {
            return;
        }
        $base = plugin_dir_url(__FILE__) . 'assets/';
        wp_enqueue_style(self::HANDLE_CSS, $base . 'font-resizer.css', [], '1.0.0');
        wp_enqueue_script(self::HANDLE_JS, $base . 'font-resizer.js', [], '1.0.0', true);

        // Pass a couple of safe selectors and the post ID
        wp_localize_script(self::HANDLE_JS, 'AH_FONT_RESIZER', [
            'postId'    => get_the_ID(),
            // Try common content containers first; fall back to our injected wrapper
            'selectors' => [
               '#ah-post-content',        // <- put wrapper first
            ],
            // Default scale keys to CSS variables map
            'scales'    => [
                'normal' => '1.0',
                'large'  => '1.5',
                'xlarge' => '2.0'
            ],
            'storageKey' => 'ah_font_scale' // site-wide; change to per-post by appending postId in JS if you prefer
        ]);
    }

    /**
     * Wrap the rendered content with a container we control, so we always have a reliable target.
     */
    public function wrap_post_content($content) {
        if (!is_singular() || is_admin() || wp_doing_ajax() || defined('REST_REQUEST')) {
            return $content;
        }

        // Avoid double-wrapping if theme already has our wrapper (rare)
        if (strpos($content, 'id="ah-post-content"') !== false) {
            return $content;
        }

        // Keep markup minimal; the scaling is applied via CSS variable on this element
        return '<div id="ah-post-content" class="ah-post-content-wrapper">' . $content . '</div>';
    }
}

/**
 * Front-end Widget: three buttons (Normal / Large / X-Large) with size indicators.
 */
class AppHaven_Font_Resizer_Widget extends WP_Widget {
    public function __construct() {
        parent::__construct(
            'apphaven_font_resizer_widget',
            __('AppHaven Font Resizer (AppHaven)', 'apphaven-font-resizer'),
            ['description' => __('Buttons to resize the current post/page text with visible size indicators.', 'apphaven-font-resizer')]
        );
    }

    public function widget($args, $instance) {
        echo $args['before_widget'];

        $title = !empty($instance['title']) ? $instance['title'] : __('Text Size', 'apphaven-font-resizer');
        if (!empty($title)) {
            echo $args['before_title'] . esc_html($title) . $args['after_title'];
        }

        // ARIA: toolbar with three toggle buttons; JS will manage aria-pressed
        echo '<div class="ah-fr-widget">';
        ?>
        <div class="ah-fr-toolbar" role="toolbar" aria-label="<?php echo esc_attr__('Font size controls', 'apphaven-font-resizer'); ?>">
            <button class="ah-fr-btn" data-ah-scale="normal" aria-pressed="false" aria-label="<?php esc_attr_e('Normal text size', 'apphaven-font-resizer'); ?>">
                <span class="ah-fr-A ah-fr-A--sm" aria-hidden="true">A</span>
                <span class="ah-fr-label"><?php esc_html_e('Normal', 'apphaven-font-resizer'); ?></span>
            </button>
            <button class="ah-fr-btn" data-ah-scale="large" aria-pressed="false" aria-label="<?php esc_attr_e('Large text size', 'apphaven-font-resizer'); ?>">
                <span class="ah-fr-A ah-fr-A--md" aria-hidden="true">A</span>
                <span class="ah-fr-label"><?php esc_html_e('Large', 'apphaven-font-resizer'); ?></span>
            </button>
            <button class="ah-fr-btn" data-ah-scale="xlarge" aria-pressed="false" aria-label="<?php esc_attr_e('Extra large text size', 'apphaven-font-resizer'); ?>">
                <span class="ah-fr-A ah-fr-A--lg" aria-hidden="true">A</span>
                <span class="ah-fr-label"><?php esc_html_e('X-Large', 'apphaven-font-resizer'); ?></span>
            </button>
        </div>
        <?php
        echo '</div>';
        echo $args['after_widget'];
    }

    public function form($instance) {
        $title = isset($instance['title']) ? $instance['title'] : __('Text Size', 'apphaven-font-resizer'); ?>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('title')); ?>">
                <?php esc_html_e('Title:', 'apphaven-font-resizer'); ?>
            </label>
            <input class="widefat"
                   id="<?php echo esc_attr($this->get_field_id('title')); ?>"
                   name="<?php echo esc_attr($this->get_field_name('title')); ?>"
                   type="text"
                   value="<?php echo esc_attr($title); ?>">
        </p>
        <?php
    }

    public function update($new_instance, $old_instance) {
        $instance          = [];
        $instance['title'] = sanitize_text_field($new_instance['title'] ?? '');
        return $instance;
    }
}

	// Helper: get plugin basename once
	function ah_fr_plugin_basename() {
		return plugin_basename(__FILE__);
	}

	/**
	 * Add links like "Settings" on the Installed Plugins screen (right under the plugin name).
	 */
	add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'ah_fr_plugin_action_links');
	function ah_fr_plugin_action_links($links) {
		// If you create a real settings page later, update this URL.
		$settings_url = admin_url('options-general.php?page=apphaven-font-resizer');

		$settings_link = sprintf(
			'<a href="%s">%s</a>',
			esc_url($settings_url),
			esc_html__('Settings', 'apphaven-font-resizer')
		);

		// Prepend Settings so it appears first
		array_unshift($links, $settings_link);

		return $links;
	}

	/**
	 * Add extra row meta: Docs, Support, etc. (small links under description).
	 */
	add_filter('plugin_row_meta', 'ah_fr_plugin_row_meta', 10, 2);
	function ah_fr_plugin_row_meta($links, $file) {
		if ($file !== plugin_basename(__FILE__)) {
			return $links;
		}

		$docs_link = sprintf(
			'<a href="%s" target="_blank" rel="noopener noreferrer">%s</a>',
			esc_url('https://apphaven.net/plugins/apphaven-font-resizer/docs'),
			esc_html__('Documentation', 'apphaven-font-resizer')
		);

		$support_link = sprintf(
			'<a href="%s" target="_blank" rel="noopener noreferrer">%s</a>',
			esc_url('https://apphaven.net/contact'),
			esc_html__('Support', 'apphaven-font-resizer')
		);

		$more_link = sprintf(
			'<a href="%s" target="_blank" rel="noopener noreferrer">%s</a>',
			esc_url('https://apphaven.net/plugins'),
			esc_html__('More plugins by AppHaven', 'apphaven-font-resizer')
		);

		// Append custom meta links
		$links[] = $docs_link;
		$links[] = $support_link;
		$links[] = $more_link;

		return $links;
	}
new AppHaven_Font_Resizer();
